%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: read_picarro_n2o
%
% Scope:    import PICARRO G5131-i data file
%
% Input:    file name G5131-i data file (string),
%           directory G5131-i data file (string),
%           index plot requested  (double)
%
% Output:   "data" struture containing raw data from G5131-i,
%           overview figure (if requested)   
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function data = read_picarro_n2o(filename_n2o,pathname_n2o, plot_requested)

tic;fprintf('Running function read_picarro_n2o...\n');

datatable = readtable([pathname_n2o filename_n2o]);

data.all(:,1) = datatable.('FRAC_HRS_SINCE_JAN1');     %absolute time in h
data.all(:,2) = data.all(:,1)-data.all(1,1);         %relative time in h
data.all(:,3) = datatable.('N2O');  
data.all(:,4) = datatable.('d15N');  
data.all(:,5) = datatable.('d15Nalpha');  
data.all(:,6) = datatable.('d15Nbeta');  
data.all(:,7) = datatable.('d18O');  
data.all(:,8) = datatable.('CavityPressure');  
data.all(:,9) = datatable.('CavityTemp');  
data.all(:,10) = datatable.('ChillerTemp');  
data.all(:,11) = datatable.('P9Temp');  
data.all(:,12) = datatable.('MPVPosition');  

data.start_time=data.all(1,1);
data.end_time=data.all(end,1);

date_string=datatable.('DATE');
time_string=datatable.('TIME');

data.start_time_string=join([string(date_string(1,1)) string(time_string(1,1))]);
data.end_time_string=join([string(date_string(end,1)) string(time_string(end,1))]);

%% define variable names for plot labels (with units)

data.varlabels ={join(['Time [h] since',extractBefore(data.start_time_string,5),'-01-01 00:00:00']),...
                join(['Time [h] since',extractBefore(data.start_time_string,20)]),...
                'N_2O [ppb]',...
                ['{\it\delta}^{15}N [',char(8240),']'],...        
                ['{\it\delta}^{15}N^{\alpha} [',char(8240),']'],...        
                ['{\it\delta}^{15}N^{\beta} [',char(8240),']'],...
                ['{\it\delta}^{18}O [',char(8240),']'],...
                '{\itp}_{Cell} [mTorr]',...
                '{\itT}_{Cell} [°C]',...
                '{\itT}_{Chiller} [°C]',...
                '{\itT}_{P9} [°C]',...
                'MPV Position'};

%% find interval boundaries (i.e. MPV position changes)

data.boundaries_idx = find(ischange(data.all(:,12))==1);
data.boundaries_idx = transpose([1 transpose([data.boundaries_idx]) height(data.all)]);

%% calculate mean time resolution

delta_t = diff(data.all(:,2))*60*60;    %s  
data.time_res=mean(delta_t);       

%% remove MPV "transition" switches (i.e. MPV position changes closer in time than 20 s)

discard_idx=round(20/data.time_res);

 check_boundaries=[];
 for i=1:height(data.boundaries_idx)-1
    if data.boundaries_idx(i+1,1)-data.boundaries_idx(i,1)<discard_idx
        check_boundaries=[check_boundaries find(data.boundaries_idx==data.boundaries_idx(i+1,1))];
    end
 end
 
 data.boundaries_idx(check_boundaries,1)=NaN;
 data.boundaries_idx(end,1)=height(data.all);
 data.boundaries_idx=rmmissing(data.boundaries_idx);

%% plot overview

plotvars=[3 7 5 8 6 11];
if plot_requested
figure;
    for i=1:6
        subplot(3,2,i);
        plot(data.all(:,2),data.all(:,plotvars(1,i)),'LineWidth',1.5);
        hold on;
        for j=2:height(data.boundaries_idx)-1
            xline(data.all(data.boundaries_idx(j,1),2),'--k');
        end
        %xline(data.all(data.boundaries_idx(2:height(data.boundaries_idx)-1,1),2),'--k');   %only works in Matlab R2021 and later versions       
        grid on;
        box on;
        xlim([0 data.all(end,2)]);
        ylabel({sprintf('%s',data.varlabels{1,plotvars(1,i)})},'fontsize',12);
        if i==1
        title('Overview PICARRO G5131-{\iti}','fontsize',12); end    
        if i==5 || i==6
        xlabel({sprintf('%s',data.varlabels{1,2})},'fontsize',12); end
    end
    
    x0 = 10;
    y0 = 50;
    ww = 900;
    hh = 700;
    set(gcf,'units','points','position',[x0,y0,ww,hh]);
    set(gcf,'Units','Inches');
    pos = get(gcf,'Position');
    set(gcf,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3),pos(4)]);
end
%print(gcf,'C:\Users\brsi\Desktop\overview_N2O','-dpng', '-r800');

%% create "flag" vector

for k=1:height(data.boundaries_idx)-1
    num=0;sum=0;
    for i=data.boundaries_idx(k,1):data.boundaries_idx(k+1,1)
        num=num+1;
        sum=sum+data.all(i,12);
    end
    data.flag(k,1)=round(sum/num);
end

%%
time_elapsed=toc; fprintf('read_picarro_n2o completed (execution time: %1.2f s)\n',time_elapsed); 

end